/******************************************************************************* 
 * Freescale Semiconductor Inc. 
 * (c) Copyright 2004-2005 Freescale Semiconductor, Inc. 
 * ALL RIGHTS RESERVED. 
 ******************************************************************************** 
 Services performed by FREESCALE in this matter are performed AS IS and without  
 any warranty. CUSTOMER retains the final decision relative to the total design  
 and functionality of the end product. FREESCALE neither guarantees nor will be  
 held liable by CUSTOMER for the success of this project. 
 FREESCALE DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,  
 BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR  
 A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO A PROJECT 
 BY FREESCALE, AND OR NAY PRODUCT RESULTING FROM FREESCALE SERVICES. IN NO EVENT 
 SHALL FREESCALE BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF 
 THIS AGREEMENT. 
 
 CUSTOMER agrees to hold FREESCALE harmless against any and all claims demands or 
 actions by anyone on account of any damage, or injury, whether commercial,  
 contractual, or tortuous, rising directly or indirectly as a result of an advise 
 or assistance supplied CUSTOMER in connection with product, services or goods  
 supplied under this Agreement. 
 ******************************************************************************** 
 * File      USB2CAN.c 
 * Owner     b32543
 * Version   0.1   
 * Date      10-21-2011
 * Classification:Internal Use Only 
 * Proprietary 
 * Brief     Brief description of the file 
 ******************************************************************************** 
 * Detailed Description: 
 ******************************************************************************** 
 Revision History: 
 Version    Date    Author    Description of Changes 
 *******************************************************************************/
#include "config.h"
#include "mcu.h"
#include "msCANdrv.h"
#include "virtual_com.h"
#include "USB2CAN.h"
#include "stdio.h"
/******************************************************************************* 
 * External objects 
 *******************************************************************************/

/******************************************************************************* 
 * Global variables 
 *******************************************************************************/
PC_CMD_TYPE current_pc_cmd;
uint_8 g_error_status;
/******************************************************************************* 
 * Constants and macros 
 *******************************************************************************/

/******************************************************************************* 
 * Local types 
 *******************************************************************************/

/******************************************************************************* 
 * Local function prototypes 
 *******************************************************************************/
static uint_8 CANMassEraseTarget(void);
static uint_8 CANSectorEraseTarget(const uint_8 *pStartAddr);
static uint_8 CANUnsecTarget(const uint_8 *pBackdoorKey, uint_8 *pUnsecResult);
static uint_8 CANBlankCheckTartget(uint_8 *pCheckResult);
static uint_8 CANBurstProgTarget(const uint_8 *pStartAddr,
        const uint_8 *pDataLenth, const uint_8 *pProgData);
static uint_8 CANByteProgTarget(const uint_8 *pProgAddr,
        const uint_8 *pProgData);
static uint_8 CANVerifyTarget(void);
static uint_8 CANReadTarget(const uint_8 *pStartAddr, const uint_8 *pDataLenth,
        uint_8 *pReadData);
static uint_8 VcomSendData(uint_8 u8Command, const uint_8* pData,
        uint_8 *pSendBuf, uint_8 u8DataLenth);
static uint_8 VcomCMDParse(PC_CMD_TYPE *pPcCMD);
static uint_8 PcDataUnpack(const uint_8 pc_data[], PC_CMD_TYPE *pPcCMD);
static void CANMBsConfig(void);
/******************************************************************************* 
 * Local variables 
 *******************************************************************************/

/******************************************************************************* 
 * Local functions 
 *******************************************************************************/

/******************************************************************************* 
 * Global functions 
 ******************************************************************************/
/******************************************************************************* 
 Function Name : USB2CAN_Init 
 Engineer  : b32543 
 Date    : 11/16/11 
 Parameters  :     
 Modifies  : NONE 
 Returns  :  
 Notes    : Initialize USB2CAN layer  
 Issues   : NONE 
 ******************************************************************************/
void USB2CAN_Init()
{
    CANMBsConfig(); /* config CAN message buffer */
}

/******************************************************************************* 
 Function Name : USB2CAN_Task 
 Engineer  : b32543 
 Date    : 11/16/2011 
 Parameters  :    
 Modifies  : NONE 
 Returns  :  None
 Notes    :  USB2CAN task: parse the pc command, then send CAN frame to target.
 Issues   : NONE 
 ******************************************************************************/
uint_8 USB2CAN_Task()
{
    uint_8 u8Status = 0U, u8DataBuff[DATA_BUFF_SIZE];
    if (g_active_usb2can_flag)
    {
        //VcomSendData(BDG_ACK_CMD, &u8Status, g_vcom_send_buf, 1U);
        /*transfer the received data to local buffer */
        for (u8Status = 0U; u8Status < g_vcom_recv_size; u8Status++)
        {
            u8DataBuff[u8Status] = g_vcom_recv_buf[u8Status];
        }
        /* parse the received data */
        g_error_status = PcDataUnpack(u8DataBuff, &current_pc_cmd);
        if (g_error_status == ERR_OK)
        {
            u8Status = VcomCMDParse(&current_pc_cmd);
        }
        else
        {
            VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
        }
        g_active_usb2can_flag = 0U; /* clear the flag to allow next operation*/
        return TRUE;
    }
    return FALSE;
}
/******************************************************************************* 
 Function Name : PcDataUnpack 
 Engineer  : b32543 
 Date    : 10/24/11 
 Parameters  : pc_data  the data from PC
 PC_CMD_TYPE *pPcCMD: command structure for storing data parsed.    
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : Function will unpack the pc data to a command structure.  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 PcDataUnpack(const uint_8 pc_data[], PC_CMD_TYPE *pPcCMD)
{
    uint_8 u8temp = 0U;
    uint_8 u8checksum = 0U;
    /* check the packet Header is correct */
    if ((pc_data[0] != PC_PACKET_HEADER) || (pc_data[1] != PC_PACKET_HEADER)
            || (pc_data[2] != PC_PACKET_HEADER) || (pc_data[3]
            != PC_PACKET_HEADER))
    {
        return FRAMEING_ERR;
    }
    /* get data length */
    u8temp = pc_data[5];
    pPcCMD->data_length = u8temp;
    u8checksum = u8temp;
    u8temp += 11U;
    /* check the packet ends are correct */
    if ((pc_data[u8temp - 1] != PC_PACKET_ENDER) || (pc_data[u8temp - 2]
            != PC_PACKET_ENDER) || (pc_data[u8temp - 3] != PC_PACKET_ENDER)
            || (pc_data[u8temp - 4] != PC_PACKET_ENDER))
    {
        return FRAMEING_ERR;
    }
    /* get command */
    pPcCMD->pc_cmd = pc_data[4];
    /* get data */
    for (u8temp = 0U; u8temp < pPcCMD->data_length; u8temp++)
    {
        pPcCMD->data[u8temp] = pc_data[6 + u8temp];
        u8checksum += pc_data[6 + u8temp];
    }
    /* get checksum */
    pPcCMD->checksum = pc_data[6 + u8temp];

    if ((pPcCMD->checksum + u8checksum) != 0xffU)
    {
        return CHECKSUM_ERR;
    }
    return ERR_OK;
}

/******************************************************************************* 
 Function Name : VcomCMDParse 
 Engineer  : b32543 
 Date    : 10/24/11 
 Parameters  :PC_CMD_TYPE *pPcCMD: pc command    
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : Function will parse pc command  
 Issues   : NONE 
 ******************************************************************************/
uint_8 VcomCMDParse(PC_CMD_TYPE *pPcCMD)
{
    uint_8 u8temp = 0U, u8Result = 0U;
    uint_32 u32Datalenth = 0U;
    uint_8 u8DataTemp[DATA_BUFF_SIZE];

    /* Parse the command */
    switch (pPcCMD->pc_cmd)
    {
        case PC_SET_CAN_BR:
            g_error_status = VcomSendData(BDG_ACK_CMD, &u8Result,
                                          g_vcom_send_buf, 1U);
            break;
        case PC_MASS_ERASE_TG:
            g_error_status = CANMassEraseTarget();
            VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
            break;
        case PC_SECTOR_ERASE_TG:
            g_error_status = CANSectorEraseTarget(&pPcCMD->data[0]);
            VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);

            break;
        case PC_BLK_CHK_TG:
            g_error_status = CANBlankCheckTartget(&u8Result);
            if (g_error_status == ERR_OK)
            {
                VcomSendData(BDG_SEND_BLKCHK_RESULT, &u8Result,
                             g_vcom_send_buf, 1U);
            }
            else
            {
                VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
            }
            break;
        case PC_PROG_TG:
            u32Datalenth = pPcCMD->data_length - 4U;
            g_error_status
                    = CANBurstProgTarget(&pPcCMD->data[0], &u32Datalenth,
                                         &pPcCMD->data[4]);
            VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
            break;
        case PC_VERIFY_TG:
            break;
        case PC_READ_TG:
            u32Datalenth = pPcCMD->data_length - 4U;
            g_error_status = CANReadTarget(&pPcCMD->data[0], &pPcCMD->data[4],
                                           u8DataTemp);
            if (g_error_status == ERR_OK)
            {
                VcomSendData(BDG_SEND_FLASH_CONTENT, u8DataTemp,
                             g_vcom_send_buf, pPcCMD->data[7]);
            }
            else
            {
                VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
            }
            break;
        case PC_UNSECURE_TG:
            g_error_status = CANUnsecTarget(&pPcCMD->data[0], &u8Result);
            if (g_error_status == TRUE)
            {
                VcomSendData(BDG_SEND_UNSEC_RESULT, &u8Result, g_vcom_send_buf,
                             1U);
            }
            else
            {
                VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
            }
            break;
        default:
            g_error_status = CMD_WORD_ERR;
            VcomSendData(BDG_ACK_CMD, &g_error_status, g_vcom_send_buf, 1U);
            break;
    }
    return ERR_OK;
}

/******************************************************************************* 
 Function Name : VcomSendData 
 Engineer  : b32543 
 Date    : 11/08/11 
 Parameters  :   
 Modifies  : NONE 
 Returns  :  FALSE---Sending failed, TRUE---Sending OK.
 Notes    : Function will send data via virtual com.  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 VcomSendData(uint_8 u8Command, const uint_8* pData,
        uint_8 *pSendBuf, uint_8 u8DataLenth)
{
    uint_8 u8Index = 0U, u8Checksum = 0U;

    if (g_vcom_send_size == 0U)
    {
        *pSendBuf++ = PC_PACKET_HEADER;
        *pSendBuf++ = PC_PACKET_HEADER;
        *pSendBuf++ = PC_PACKET_HEADER;
        *pSendBuf++ = PC_PACKET_HEADER;
        *pSendBuf++ = u8Command;
        *pSendBuf++ = u8DataLenth;
        u8Checksum = u8DataLenth;
        for (u8Index = 0U; u8Index < u8DataLenth; u8Index++)
        {
            u8Checksum += *pData;
            *pSendBuf++ = *pData++;
        }
        *pSendBuf++ = (uint_8) ~(uint_8) (u8Checksum);
        *pSendBuf++ = PC_PACKET_ENDER;
        *pSendBuf++ = PC_PACKET_ENDER;
        *pSendBuf++ = PC_PACKET_ENDER;
        *pSendBuf = PC_PACKET_ENDER;
        g_vcom_send_size = u8DataLenth + 11U;
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}
/******************************************************************************* 
 Function Name : CANUnsecTarget 
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters  :const uint_8 *pBackdoorKey: Backdoor Key  
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    :  unsecure target.  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANUnsecTarget(const uint_8 *pBackdoorKey, uint_8 *pUnsecResult)
{
    uint_8 u8MBData[9], u8RecData[9], u8Index = 0U, u8OverTimeFlag = 0U;
    uint_8 u8MBStatus = 0U;

    /* transmit command to target */
    u8MBData[0] = 8U; /* 8-byte long data */
    for (u8Index = 1U; u8Index < 9U; u8Index++)
    {
        u8MBData[u8Index] = *pBackdoorKey++;
    }
    CAN_LoadMB(CMD_SEND_MB(BDG_UNSEC_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_UNSEC_TG));
    /* receive target's response */
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_UNSEC_TG),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_UNSEC_TG),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        *pUnsecResult = 0x00U; /* ack error */
        return TIMEOUT_ERR;
    }
    *pUnsecResult = u8RecData[1]; /* ack error */
    return ERR_OK;
}

/******************************************************************************* 
 Function Name : CANMassEraseTarget 
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters  :None 
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : Mass erase target.   
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANMassEraseTarget()
{
    uint_8 u8MBData[9], u8RecData[9], u8Index = 0U, u8OverTimeFlag = 0U;
    uint_8 u8MBStatus = 0U;
    /* transmit command to target */
    u8MBData[0] = 0U; /* no data */

    CAN_LoadMB(CMD_SEND_MB(BDG_MASS_ERASE_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_MASS_ERASE_TG));
    /* receive target's response */
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_MASS_ERASE_TG),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_MASS_ERASE_TG),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        return TIMEOUT_ERR;
    }

    return ERR_OK;
}

/******************************************************************************* 
 Function Name : CANSectorEraseTarget
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters  :const uint_8 *pSectorAddr: pointer to the sector address.    
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    :  Erase the flash sector specified. 
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANSectorEraseTarget(const uint_8 *pSectorAddr)
{
    uint_8 u8MBData[9], u8RecData[9], u8Index = 0U, u8OverTimeFlag = 0U;
    uint_8 u8MBStatus = 0U;
    /* transmit command to target */
    u8MBData[0] = 4U; /* 8-byte long data */
    for (u8Index = 1U; u8Index < 5U; u8Index++)
    {
        u8MBData[u8Index] = *pSectorAddr++;
    }
    CAN_LoadMB(CMD_SEND_MB(BDG_SECT_ERASE_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_SECT_ERASE_TG));
    /* receive target's response */
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_SECT_ERASE_TG),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_SECT_ERASE_TG),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        return TIMEOUT_ERR;
    }
    /* compare the data received with the data sent */
    for (u8Index = 0U; u8Index < 5U; u8Index++)
    {
        if(u8MBData[u8Index ] != u8RecData[u8Index])
        {
            return RESPONSE_DATA_ERR;
        }
    }
    return ERR_OK;
}

/******************************************************************************* 
 Function Name : CANBlankCheckTartget 
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters  :  uint_8 *pCheckResult: pointer to the blank check result.   
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    :  blank check target.  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANBlankCheckTartget(uint_8 *pCheckResult)
{
    uint_8 u8OverTimeFlag = 0U;
    uint_8 u8MBData[9], u8RecData[9], u8Index = 0U;
    uint_8 u8MBStatus = 0U;
    /* transmit command to target */
    u8MBData[0] = 0U; /* 8-byte long data */
    CAN_LoadMB(CMD_SEND_MB(BDG_BLANK_CHECK_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_BLANK_CHECK_TG));
    /* receive target's response */
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_BLANK_CHECK_TG),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_BLANK_CHECK_TG),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        *pCheckResult = 0x00U;
        return TIMEOUT_ERR;
    }
    /* assign result to the pointer */
    *pCheckResult = u8RecData[1];
    return ERR_OK;
}
/******************************************************************************* 
 Function Name : CANBurstProgTarget 
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters : const uint_8 *pStartAddr,
 : const uint_8 *pDataLength, 
 : const uint_8 *pProgData  
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : Burst Program flash of target.  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANBurstProgTarget(const uint_8 *pStartAddr,
        const uint_8 *pDataLength, const uint_8 *pProgData)
{
    uint_8 u8OverTimeFlag = 0U;
    uint_8 u8MBData[9], u8RecData[9], u8Index = 0U;
    uint_32 u32DataLenth = *(uint_32 *) pDataLength;

    uint_8 u8MBStatus = 0U;
    /*SET_ADDR command: set the start address and data length */
    u8MBData[0] = 8U; /* 8-byte data */
    for (u8Index = 0U; u8Index < 4U; u8Index++)
    {
        u8MBData[u8Index + 1U] = *pStartAddr++;
        u8MBData[u8Index + 5U] = *pDataLength++;
    }
    CAN_LoadMB(CMD_SEND_MB(BDG_SET_ADDR), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_SET_ADDR));
    /* receive target's response */
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_SET_ADDR),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_SET_ADDR),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        return TIMEOUT_ERR;
    }
    /* compare the data received with the data sent */
    for (u8Index = 0U; u8Index < 6U; u8Index++)
    {
        if(u8MBData[u8Index] != u8RecData[u8Index])
        {
            return RESPONSE_DATA_ERR;
        }
    }
    /* LOAD_DATA */
    /* got data length, be sure the data is big endian order */

    while (u32DataLenth)
    {
        if(u32DataLenth < 9U)
        {
            u8MBData[0] = (uint_8)u32DataLenth;
            for(u8Index = 0U; u8Index < u32DataLenth; u8Index++)
            {
                u8MBData[u8Index + 1U] = *pProgData++;
            }
            CAN_LoadMB(CMD_SEND_MB(BDG_LOAD_DATA),u8MBData);
            CAN_TransmitMB(CMD_SEND_MB(BDG_LOAD_DATA));
            u32DataLenth = 0U;
            /* receive target's response */
            START_OVERTIMER_CNT;
            while(1)
            {
                __RESET_WATCHDOG();
                if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_LOAD_DATA),&u8MBStatus) == ERR_OK)
                {
                    if(u8MBStatus == NEWDATA )
                    {
                        CAN_ReadDataMB(RESPONSE_REC_MB(BDG_LOAD_DATA),u8RecData);
                        u8OverTimeFlag = 0U;
                        break;
                    }
                }
                /* over time manage */
                if(OVERTIMER_FLAG == TRUE)
                {
                    STOP_OVERTIMER_CNT;
                    CLEAR_OVTM_FLG;
                    u8OverTimeFlag = 1U;
                    break;
                }
            }
            STOP_OVERTIMER_CNT;
            if (u8OverTimeFlag)
            {
                return TIMEOUT_ERR;
            }
            /* compare the data received with the data sent */
            for (u8Index = 0U; u8Index < u8MBData[0]; u8Index++)
            {
                if(u8MBData[u8Index] != u8RecData[u8Index])
                {
                    return RESPONSE_DATA_ERR;
                }
            }
        }
        else
        {
            u8MBData[0] = 8U;
            for(u8Index = 0U; u8Index < 8U; u8Index++)
            {
                u8MBData[u8Index + 1U] = *pProgData++;
            }
            CAN_LoadMB(CMD_SEND_MB(BDG_LOAD_DATA),u8MBData);
            CAN_TransmitMB(CMD_SEND_MB(BDG_LOAD_DATA));
            u32DataLenth -= 8U;
            /* receive target's response */
            START_OVERTIMER_CNT;
            while(1)
            {
                __RESET_WATCHDOG();
                if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_LOAD_DATA),&u8MBStatus) == ERR_OK)
                {
                    if(u8MBStatus == NEWDATA )
                    {
                        CAN_ReadDataMB(RESPONSE_REC_MB(BDG_LOAD_DATA),u8RecData);
                        u8OverTimeFlag = 0U;
                        break;
                    }
                }
                /* over time manage */
                if(OVERTIMER_FLAG == TRUE)
                {
                    STOP_OVERTIMER_CNT;
                    CLEAR_OVTM_FLG;
                    u8OverTimeFlag = 1U;
                    break;
                }
            }
            STOP_OVERTIMER_CNT;
            if (u8OverTimeFlag)
            {
                return TIMEOUT_ERR;
            }
            /* compare the data received with the data sent */
            for (u8Index = 0U; u8Index < 9U; u8Index++)
            {
                if(u8MBData[u8Index] != u8RecData[u8Index])
                {
                    return RESPONSE_DATA_ERR;
                }
            }
        }
    }
    /* send burst program command */
    u8MBData[0] = 0U;
    CAN_LoadMB(CMD_SEND_MB(BDG_BURST_PROG_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_BURST_PROG_TG));
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_BURST_PROG_TG),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_BURST_PROG_TG),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        return TIMEOUT_ERR;
    }

    return ERR_OK;
}

/******************************************************************************* 
 Function Name : CANByteProgTarget 
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters :  const uint_8 *pProgAddr, const uint_8 *pProgData
 Modifies  : 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : program the target flash by byte.  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANByteProgTarget(const uint_8 *pProgAddr,
        const uint_8 *pProgData)
{
    uint_8 u8OverTimeFlag = 0U;
    uint_8 u8MBData[9], u8RecData[8], u8Index = 0U;
    uint_8 u8MBStatus = 0U;
    /* transmit command to target */
    u8MBData[0] = 5U; /* 8-byte long data */
    for (u8Index = 1U; u8Index < 5U; u8Index++)
    {
        u8MBData[u8Index] = *pProgAddr++;
    }
    u8MBData[5] = *pProgData;
    CAN_LoadMB(CMD_SEND_MB(BDG_BYTE_PROG_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_BYTE_PROG_TG));
    /* receive target's response */
    START_OVERTIMER_CNT;
    while (1)
    {
        __RESET_WATCHDOG();
        if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_BYTE_PROG_TG),&u8MBStatus) == ERR_OK)
        {
            if(u8MBStatus == NEWDATA )
            {
                CAN_ReadDataMB(RESPONSE_REC_MB(BDG_BYTE_PROG_TG),u8RecData);
                u8OverTimeFlag = 0U;
                break;
            }
        }
        /* over time manage */
        if(OVERTIMER_FLAG == TRUE)
        {
            STOP_OVERTIMER_CNT;
            CLEAR_OVTM_FLG;
            u8OverTimeFlag = 1U;
            break;
        }
    }
    STOP_OVERTIMER_CNT;
    if (u8OverTimeFlag)
    {
        return TIMEOUT_ERR;
    }
    /* compare the data received with the data sent */
    for (u8Index = 0U; u8Index < 6U; u8Index++)
    {
        if(u8MBData[u8Index] != u8RecData[u8Index])
        {
            return RESPONSE_DATA_ERR;
        }
    }
    return ERR_OK;
}
/******************************************************************************* 
 Function Name : VcomCMDParse 
 Engineer  : b32543 
 Date    : 10/24/11 
 Parameters  :PC_CMD_TYPE *pPcCMD: pc command    
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : Function will parse pc command  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANVerifyTarget()
{

}
/******************************************************************************* 
 Function Name : CANReadTarget 
 Engineer  : b32543 
 Date    : 11/02/11 
 Parameters  : const uint_8 *pStartAddr
 : const uint_8 *pDataLenth,
 : uint_8 *pReadData   
 Modifies  : NONE 
 Returns  :  FALSE---operation failed, TRUE---operation OK. 
 Notes    : Function will parse pc command  
 Issues   : NONE 
 ******************************************************************************/
static uint_8 CANReadTarget(const uint_8 *pStartAddr, const uint_8 *pDataLenth,
        uint_8 *pReadData)
{
    uint_8 u8OverTimeFlag = 0U, u8MBStatus = 0U;
    uint_8 u8MBData[9], u8RecData[9], u8Index = 0U;
    uint_32 u32DataLenth = 0U;
    /* receive data fed back from target */
    u32DataLenth = *(uint_32 *) pDataLenth;
    /* send read target command. */
    u8MBData[0] = 8U;
    for (u8Index = 0U; u8Index < 4U; u8Index++)
    {
        u8MBData[u8Index + 1U] = *pStartAddr++;
        u8MBData[u8Index + 5U] = *pDataLenth++;
    }
    CAN_LoadMB(CMD_SEND_MB(BDG_READ_TG), u8MBData);
    CAN_TransmitMB(CMD_SEND_MB(BDG_READ_TG));

    while (u32DataLenth)
    {
        START_OVERTIMER_CNT;
        while(1)
        {
            __RESET_WATCHDOG();
            if(CAN_CheckStatusMB(RESPONSE_REC_MB(BDG_READ_TG),&u8MBStatus) == ERR_OK)
            {
                if(u8MBStatus == NEWDATA )
                {
                    CAN_ReadDataMB(RESPONSE_REC_MB(BDG_READ_TG),u8RecData);
                    u8OverTimeFlag = 0U;
                    break;
                }
            }
            /* over time manage */
            if(OVERTIMER_FLAG == TRUE)
            {
                STOP_OVERTIMER_CNT;
                CLEAR_OVTM_FLG;
                u8OverTimeFlag = 1U;
                break;
            }
        }
        STOP_OVERTIMER_CNT;
        if(u8OverTimeFlag)
        {
            return TIMEOUT_ERR;
        }
        if(u32DataLenth > 8U)
        {

            for(u8Index = 0U; u8Index < 8U; u8Index++)
            {
                *pReadData++ = u8RecData[u8Index + 1U];
            }
            u32DataLenth -= 8U;

        }
        /* store the last bytes of data */
        else
        {
            for(u8Index = 0U; u8Index < u32DataLenth; u8Index++)
            {
                *pReadData++ = u8RecData[u8Index + 1U];
            }
            u32DataLenth = 0U;
        }

    }
    return ERR_OK;
}

/******************************************************************************* 
 Function Name : CANMBsConfig 
 Engineer  : b32543 
 Date    : 11/1/11 
 Parameters  :void    
 Modifies  : NONE 
 Returns  :  void 
 Notes    : configure all CAN message buffers  
 Issues   : NONE 
 ******************************************************************************/
static void CANMBsConfig(void)
{
    uint_8 u8Index = 0U;

    /* MB0-15 are used to receive feedback from target */
    /* MB16-31 are used to transmit can frame to target */
    for (u8Index = 0U; u8Index < 9U; u8Index++)
    {
        CAN_ConfigMB(u8Index,RXDF,u8Index);
        CAN_ConfigMB((u8Index + 16U),TXDF,(u8Index + 16U));
    }
}

/*End file*/
